/*
 * Decompiled with CFR 0.152.
 */
package chatty;

import chatty.Irc;
import chatty.Logging;
import chatty.util.DateTime;
import chatty.util.RingBuffer;
import chatty.util.StringUtil;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.util.LinkedList;
import java.util.logging.Logger;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;

public class Connection
implements Runnable {
    private static final Logger LOGGER = Logger.getLogger(Connection.class.getName());
    private final InetSocketAddress address;
    private final Irc irc;
    private final RingBuffer<Msg> debugBuffer = new RingBuffer(20);
    private int debugCounter = -1;
    private Socket socket;
    private PrintWriter out;
    private BufferedReader in;
    private boolean connected = false;
    private int disconnectReason = -1;
    private String disconnectMessage = null;
    private int connectionCheckedCount;
    private static final int CONNECT_TIMEOUT = 10000;
    private static final int SOCKET_BLOCK_TIMEOUT = 15000;
    private static final int PING_AFTER_CHECKS = 3;
    private final String idPrefix;
    private final boolean secured;

    public Connection(Irc irc, InetSocketAddress address, String id, boolean secured) {
        this.irc = irc;
        this.address = address;
        this.idPrefix = "[" + id + "] ";
        this.secured = secured;
    }

    private final void info(String message) {
        LOGGER.info(this.idPrefix + message);
    }

    private final void warning(String message) {
        LOGGER.warning(this.idPrefix + message);
    }

    public InetSocketAddress getAddress() {
        return this.address;
    }

    @Override
    public void run() {
        Charset charset = Charset.forName("UTF-8");
        try {
            this.info("Trying to connect to " + this.address + (this.secured ? " (secured)" : ""));
            if (this.secured) {
                SSLSocketFactory sf = (SSLSocketFactory)SSLSocketFactory.getDefault();
                this.socket = sf.createSocket();
                ((SSLSocket)this.socket).setUseClientMode(true);
            } else {
                this.socket = new Socket();
            }
            this.socket.connect(this.address, 10000);
            this.out = new PrintWriter(new OutputStreamWriter(this.socket.getOutputStream(), charset));
            this.in = new BufferedReader(new InputStreamReader(this.socket.getInputStream(), charset));
            this.socket.setSoTimeout(15000);
        }
        catch (UnknownHostException ex) {
            this.irc.disconnected(100);
            this.warning("Error opening connection to " + this.address + ": " + ex);
            return;
        }
        catch (SocketTimeoutException ex) {
            this.warning("Error opening connection: " + ex);
            this.irc.disconnected(101);
            return;
        }
        catch (IOException ex) {
            this.warning("Error opening connection: " + ex);
            this.irc.disconnected(102, ex.getLocalizedMessage());
            return;
        }
        this.info("Connected to " + this.socket.getRemoteSocketAddress().toString());
        this.connected = true;
        this.irc.connected(this.socket.getInetAddress().toString(), this.address.getPort());
        StringBuilder b = new StringBuilder();
        boolean previousWasCR = false;
        String receivedLine = null;
        while (true) {
            try {
                int c;
                while ((c = this.in.read()) != -1) {
                    if (c == 13) {
                        previousWasCR = true;
                    } else if (c == 10) {
                        if (previousWasCR) {
                            receivedLine = b.toString();
                            b.setLength(0);
                            previousWasCR = false;
                        }
                    } else {
                        b.append((char)c);
                        previousWasCR = false;
                    }
                    if (receivedLine == null) continue;
                    this.debugBuffer.add(new Msg(System.currentTimeMillis(), receivedLine, false));
                    this.irc.received(receivedLine);
                    receivedLine = null;
                    this.activity();
                }
            }
            catch (SocketTimeoutException ex) {
                this.checkConnection();
                continue;
            }
            catch (SSLException ex) {
                this.warning("SSL Error reading from socket: " + ex);
                this.disconnectReason = 107;
                this.disconnectMessage = ex.getLocalizedMessage();
            }
            catch (IOException ex) {
                this.info("Error reading from socket: " + ex);
            }
            break;
        }
        this.close();
    }

    private void activity() {
        this.connectionCheckedCount = 0;
        if (this.debugCounter != -1) {
            ++this.debugCounter;
            if (this.debugCounter == 10) {
                this.debug();
                this.debugCounter = -1;
            }
        }
    }

    private void checkConnection() {
        ++this.connectionCheckedCount;
        if (this.connectionCheckedCount == 3) {
            this.send("PING");
        } else if (this.connectionCheckedCount > 3) {
            LOGGER.log(Logging.USERINFO, "Warning: Server not responding");
            this.warning("No message received from server after PING for 15000ms");
            this.debugCounter = 0;
            this.connectionCheckedCount -= 6;
        }
    }

    public synchronized void close() {
        if (this.connected) {
            this.info("Closing socket.");
            try {
                this.out.close();
                this.in.close();
                this.socket.close();
            }
            catch (IOException ex) {
                this.warning("Error closing socket: " + ex);
            }
            if (this.disconnectReason == -1) {
                this.disconnectReason = 104;
            }
            if (this.disconnectMessage == null) {
                this.disconnectMessage = "";
            }
            this.irc.disconnected(this.disconnectReason, this.disconnectMessage);
        }
        this.connected = false;
    }

    public synchronized void send(String data) {
        data = StringUtil.removeLinebreakCharacters(data);
        this.debugBuffer.add(new Msg(System.currentTimeMillis(), data, true));
        this.irc.sent(data);
        this.out.print(data + "\r\n");
        this.out.flush();
    }

    public void debug() {
        LinkedList<Msg> recent = this.debugBuffer.getItems();
        StringBuilder b = new StringBuilder();
        b.append(this.idPrefix);
        b.append(this.address);
        if (this.secured) {
            b.append(" (secured)");
        }
        b.append(" / Check count: ");
        b.append(this.connectionCheckedCount).append("/").append(3);
        b.append("\n");
        for (Msg msg : recent) {
            b.append(DateTime.formatExact(msg.time));
            b.append(" ");
            if (msg.sent) {
                b.append("<<< ");
            }
            b.append(Connection.filterToken(msg.raw)).append("\n");
        }
        LOGGER.info(b.toString());
    }

    private static String filterToken(String msg) {
        if (msg.startsWith("PASS")) {
            return "PASS <token>";
        }
        return msg;
    }

    private static class Msg {
        public final long time;
        public final String raw;
        public final boolean sent;

        public Msg(long time, String raw, boolean sent) {
            this.time = time;
            this.raw = raw;
            this.sent = sent;
        }
    }
}

