/*
 * Decompiled with CFR 0.152.
 */
package chatty.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

public class BackupManager {
    private static final Logger LOGGER = Logger.getLogger(BackupManager.class.getName());
    private static final Charset CHARSET = Charset.forName("UTF-8");
    private static final String META_FILE = "backup_meta";
    private final Path defaultSourcePath;
    private final Path backupPath;
    private final List<Path> files;
    private long lastBackup;
    private int number;

    public BackupManager(Path backupPath, Path defaultSourcePath) {
        this.backupPath = backupPath;
        this.defaultSourcePath = defaultSourcePath;
        this.files = new ArrayList<Path>();
    }

    public void addFile(Path file) {
        this.files.add(file);
    }

    public void addFile(String fileName) {
        this.addFile(Paths.get(fileName, new String[0]));
    }

    public void performBackup(int delay, int count) {
        this.loadMetadata();
        if (!this.checkBackupDelay(delay * 1000) || count <= 0) {
            return;
        }
        try {
            Files.createDirectories(this.backupPath, new FileAttribute[0]);
        }
        catch (IOException ex) {
            LOGGER.warning("Could not create backup dir: " + ex.getLocalizedMessage());
            return;
        }
        this.number = (this.number + 1) % count;
        LOGGER.info("Performing backup (" + this.number + ")..");
        for (Path sourceFile : this.files) {
            if (!Files.isRegularFile(sourceFile = this.defaultSourcePath.resolve(sourceFile), new LinkOption[0])) continue;
            String targetFileName = "backup_" + this.number + "_" + sourceFile.getFileName().toString();
            Path targetFile = this.backupPath.resolve(targetFileName);
            try {
                Files.copy(sourceFile, targetFile, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException ex) {
                LOGGER.warning("Could not perform backup: " + ex);
            }
        }
        this.lastBackup = System.currentTimeMillis();
        this.saveMetadata();
    }

    private boolean checkBackupDelay(long backupDelay) {
        long ago = System.currentTimeMillis() - this.lastBackup;
        return ago > backupDelay;
    }

    private void loadMetadata() {
        Path f = this.backupPath.resolve(META_FILE);
        try (BufferedReader reader = Files.newBufferedReader(f, CHARSET);){
            String line = reader.readLine();
            String[] split = line.split(" ");
            this.number = Integer.parseInt(split[0]);
            this.lastBackup = Long.parseLong(split[1]);
        }
        catch (IOException ex) {
            LOGGER.warning("No backup meta file, using default. " + ex);
        }
        catch (ArrayIndexOutOfBoundsException | NumberFormatException ex) {
            LOGGER.warning("Backup meta file invalid format, using default. " + ex);
            this.number = 0;
            this.lastBackup = 0L;
        }
    }

    private void saveMetadata() {
        Path f = this.backupPath.resolve(META_FILE);
        try (BufferedWriter writer = Files.newBufferedWriter(f, CHARSET, new OpenOption[0]);){
            writer.write(String.valueOf(this.number) + " " + String.valueOf(this.lastBackup));
        }
        catch (IOException | NumberFormatException ex) {
            LOGGER.warning("Error writing backup meta file: " + ex);
        }
    }
}

