/*
 * Decompiled with CFR 0.152.
 */
package chatty.util;

import chatty.util.Debugging;
import chatty.util.ElapsedTime;
import chatty.util.Pair;
import chatty.util.ProcessManager;
import chatty.util.StringUtil;
import chatty.util.commands.CustomCommand;
import chatty.util.commands.Parameters;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.logging.Logger;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Clip;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.FloatControl;
import javax.sound.sampled.Line;
import javax.sound.sampled.LineEvent;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.Mixer;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.swing.SwingUtilities;
import javax.swing.Timer;

public class Sound {
    private static final int MAX_VOLUME = 100;
    private static final int MIN_VOLUME = 0;
    private static final int MIN_GAIN = -40;
    private static final Logger LOGGER = Logger.getLogger(Sound.class.getName());
    public static final Object SOUND_COMMAND_UNIQUE = new Object();
    private static Sound instance;
    private final Map<String, Long> lastPlayed = new HashMap<String, Long>();
    private final List<Pair<Clip, ElapsedTime>> clips = new LinkedList<Pair<Clip, ElapsedTime>>();
    private Mixer mixer;
    private String mixerName;
    private CustomCommand command;

    public static Sound get() {
        if (instance == null) {
            instance = new Sound();
        }
        return instance;
    }

    public static void play(Path file, float volume, String id, int delay) throws Exception {
        Sound.get().playInternal(file, volume, id, delay);
    }

    public static void play(URL file, float volume, String id, int delay) throws Exception {
        Sound.get().playInternal(file, volume, id, delay);
    }

    public static void setDeviceName(String name) {
        Sound.get().setDeviceNameInternal(name);
    }

    public static void setCommand(boolean enabled, String command) {
        Sound.get().setCommandInternal(enabled, command);
    }

    Sound() {
        Timer timer = new Timer(10000, e -> this.clearClips());
        timer.setRepeats(true);
        timer.start();
    }

    private boolean checkDelay(String id, int delay) {
        long timePassed;
        if (this.lastPlayed.containsKey(id) && (timePassed = (System.currentTimeMillis() - this.lastPlayed.get(id)) / 1000L) < (long)delay) {
            return false;
        }
        if (delay >= 0) {
            this.lastPlayed.put(id, System.currentTimeMillis());
        }
        return true;
    }

    private void playInternal(Path file, float volume, String id, int delay) throws Exception {
        if (this.checkDelay(id, delay)) {
            if (this.command != null) {
                this.runCommand(this.command, file, volume);
            } else {
                this.playInternal2(file.toUri().toURL(), volume, id);
            }
        }
    }

    private void playInternal(URL file, float volume, String id, int delay) throws Exception {
        if (this.checkDelay(id, delay)) {
            this.playInternal2(file, volume, id);
        }
    }

    private void playInternal2(URL file, float volume, String id) throws Exception {
        try {
            Clip clip = this.createClip(file, id);
            this.clips.add(new Pair<Clip, ElapsedTime>(clip, new ElapsedTime(true)));
            String volumeInfo = this.setVolume(clip, volume);
            clip.start();
            LOGGER.info(String.format("Playing[%s]: %sms %s (%s) EDT: %s", id, clip.getMicrosecondLength() / 1000L, file, volumeInfo, SwingUtilities.isEventDispatchThread()));
        }
        catch (Exception ex) {
            LOGGER.warning("Couldn't play sound (" + id + "/" + file + "): " + ex);
            throw ex;
        }
    }

    private Clip createClip(URL file, String id) throws IOException, UnsupportedAudioFileException, LineUnavailableException {
        AudioInputStream ais = AudioSystem.getAudioInputStream(file);
        ais = Sound.addConversion(ais, this.mixer != null ? this.mixer : AudioSystem.getMixer(null));
        DataLine.Info info = new DataLine.Info(Clip.class, ais.getFormat());
        Clip clip = this.mixer != null ? (Clip)this.mixer.getLine(info) : (Clip)AudioSystem.getLine(info);
        clip.open(ais);
        clip.addLineListener(event -> {
            boolean simulateIssue;
            boolean bl = simulateIssue = Debugging.isEnabled("soundNoStop") && ThreadLocalRandom.current().nextBoolean();
            if (simulateIssue && event.getType() == LineEvent.Type.STOP) {
                return;
            }
            LOGGER.info("LineEvent[" + id + "]: " + event.getType());
            if (event.getType() == LineEvent.Type.STOP) {
                clip.close();
            }
        });
        return clip;
    }

    private String setVolume(Clip clip, float volume) {
        String volumeInfo;
        FloatControl gain = Sound.getFirstAvailableControl(clip, FloatControl.Type.MASTER_GAIN, FloatControl.Type.VOLUME);
        if (gain != null) {
            gain.setValue(Sound.calculateGain(volume, gain.getMinimum(), gain.getMaximum()));
            volumeInfo = gain.toString();
        } else {
            volumeInfo = "no volume control";
        }
        return volumeInfo;
    }

    private static FloatControl getFirstAvailableControl(Clip clip, FloatControl.Type ... types) {
        for (FloatControl.Type type : types) {
            if (!clip.isControlSupported(type)) continue;
            return (FloatControl)clip.getControl(type);
        }
        return null;
    }

    private static float calculateGain(float volume, float min, float max) {
        if (volume > 100.0f) {
            volume = 100.0f;
        }
        if (volume < 0.0f) {
            volume = 0.0f;
        }
        volume = (float)(100.0 - 100.0 * Math.pow(1.25, -0.25 * (double)volume));
        if (min < -40.0f) {
            min = -40.0f;
        }
        float range = max - min;
        float gain = range * volume / 100.0f + min;
        return gain;
    }

    private static AudioInputStream addConversion(AudioInputStream ais, Mixer mixer) {
        if (Debugging.isEnabled("audio")) {
            LOGGER.info(String.format("[Input Stream Format] %s\n[Mixer] %s\n%s", ais.getFormat(), mixer.getMixerInfo(), StringUtil.join(mixer.getSourceLineInfo(), "\n")));
        }
        AudioFormat convertToFormat = null;
        for (Line.Info info : mixer.getSourceLineInfo()) {
            DataLine.Info sdi;
            if (!(info instanceof DataLine.Info) || (sdi = (DataLine.Info)info).isFormatSupported(ais.getFormat())) continue;
            Object[] formats = sdi.getFormats();
            Sound.sortFormats((AudioFormat[])formats);
            for (AudioFormat audioFormat : formats) {
                if (!AudioSystem.isConversionSupported(audioFormat, ais.getFormat())) continue;
                convertToFormat = new AudioFormat(audioFormat.getEncoding(), audioFormat.getSampleRate() == -1.0f ? ais.getFormat().getSampleRate() : audioFormat.getSampleRate(), audioFormat.getSampleSizeInBits() == -1 ? ais.getFormat().getSampleSizeInBits() : audioFormat.getSampleSizeInBits(), audioFormat.getChannels() == -1 ? ais.getFormat().getChannels() : audioFormat.getChannels(), audioFormat.getFrameSize() == -1 ? ais.getFormat().getFrameSize() : audioFormat.getFrameSize(), audioFormat.getFrameRate() == -1.0f ? ais.getFormat().getFrameRate() : audioFormat.getFrameRate(), audioFormat.isBigEndian());
                break;
            }
            if (Debugging.isEnabled("audio")) {
                LOGGER.info(String.format("[Available Formats] %s\n%s\n[Selected Format]\n%s", sdi, StringUtil.join(formats, "\n"), convertToFormat));
            }
            if (convertToFormat != null) break;
        }
        if (convertToFormat != null) {
            ais = AudioSystem.getAudioInputStream(convertToFormat, ais);
        }
        return ais;
    }

    private static void sortFormats(AudioFormat[] formats) {
        Arrays.sort(formats, new Comparator<AudioFormat>(){

            @Override
            public int compare(AudioFormat o1, AudioFormat o2) {
                if (o1.getChannels() != o2.getChannels()) {
                    return o2.getChannels() - o1.getChannels();
                }
                if (o1.getSampleSizeInBits() != o2.getSampleSizeInBits()) {
                    return o2.getSampleSizeInBits() - o1.getSampleSizeInBits();
                }
                return (int)(o2.getSampleRate() - o1.getSampleRate());
            }
        });
    }

    public static List<String> getDeviceNames() {
        ArrayList<String> result = new ArrayList<String>();
        try {
            for (Mixer.Info mixerInfo : AudioSystem.getMixerInfo()) {
                for (Line.Info lineInfo : AudioSystem.getMixer(mixerInfo).getSourceLineInfo()) {
                    if (lineInfo.getLineClass() != Clip.class) continue;
                    result.add(mixerInfo.getName());
                }
            }
        }
        catch (Exception ex) {
            LOGGER.warning("Error getting list of sound devices: " + ex);
        }
        return result;
    }

    public void setDeviceNameInternal(String name) {
        if (this.mixerName != null && this.mixerName.equals(name)) {
            return;
        }
        this.mixerName = name;
        if (name == null || name.isEmpty()) {
            this.mixer = null;
            LOGGER.info("Set to default sound device");
            return;
        }
        for (Mixer.Info info : AudioSystem.getMixerInfo()) {
            if (!info.getName().equals(name)) continue;
            this.mixer = AudioSystem.getMixer(info);
            LOGGER.info("Set sound device to " + name);
            return;
        }
        this.mixer = null;
        LOGGER.info("Could not find sound device " + name);
    }

    private void clearClips() {
        if (this.clips.isEmpty()) {
            return;
        }
        Iterator<Pair<Clip, ElapsedTime>> it = this.clips.iterator();
        int clipsClosed = 0;
        while (it.hasNext()) {
            boolean clipLengthPassed;
            Pair<Clip, ElapsedTime> entry = it.next();
            Clip clip = (Clip)entry.key;
            if (!clip.isOpen()) {
                it.remove();
                continue;
            }
            long clipLength = clip.getMicrosecondLength();
            boolean bl = clipLengthPassed = clipLength != -1L && ((ElapsedTime)entry.value).millisElapsed(clipLength / 1000L + 1000L);
            if (!clipLengthPassed && !((ElapsedTime)entry.value).secondsElapsed(120)) continue;
            ++clipsClosed;
            clip.close();
            it.remove();
        }
        if (clipsClosed > 0) {
            LOGGER.warning(String.format(Locale.ROOT, "%d clips closed which should already have been closed", clipsClosed));
        }
    }

    private void setCommandInternal(boolean enabled, String commandRaw) {
        this.command = enabled ? CustomCommand.parse(commandRaw) : null;
    }

    public String runCommand(CustomCommand command, Path file, float volume) {
        Parameters param = Parameters.create("");
        param.put("file", file.toAbsolutePath().toString().replace("\"", "\\\""));
        param.put("volume", String.valueOf(volume));
        if (command.hasError()) {
            LOGGER.warning("Sound command error: " + command.getSingleLineError());
            return "Error: " + command.getSingleLineError();
        }
        String resultCommand = command.replace(param);
        ProcessManager.execute(resultCommand, "SoundCommand", null);
        return "Running: " + resultCommand;
    }

    public static void logAudioSystemInfo() {
        StringBuilder b = new StringBuilder();
        b.append("\n=== Audio System Debug Information ===\n");
        try {
            Mixer.Info[] mixerInfos = AudioSystem.getMixerInfo();
            b.append("Total mixers: ").append(mixerInfos.length).append("\n");
            Mixer.Info defaultMixerInfo = AudioSystem.getMixer(null).getMixerInfo();
            for (int i = 0; i < mixerInfos.length; ++i) {
                Mixer.Info info = mixerInfos[i];
                b.append("Mixer ").append(i + 1);
                if (defaultMixerInfo.getName().equals(info.getName())) {
                    b.append(" (default)");
                }
                b.append(":").append("\n");
                b.append("  Name: ").append(info.getName()).append("\n");
                b.append("  Description: ").append(info.getDescription()).append("\n");
                b.append("  Vendor: ").append(info.getVendor()).append("\n");
                b.append("  Version: ").append(info.getVersion()).append("\n");
                try {
                    Mixer mixer = AudioSystem.getMixer(info);
                    Line.Info[] sourceLines = mixer.getSourceLineInfo();
                    Line.Info[] targetLines = mixer.getTargetLineInfo();
                    b.append("  Source lines: ").append(sourceLines.length).append("\n");
                    for (Line.Info lineInfo : sourceLines) {
                        b.append("    - ").append(lineInfo.getLineClass().getSimpleName());
                        b.append(" (").append(lineInfo.toString()).append(")").append("\n");
                    }
                    b.append("  Target lines: ").append(targetLines.length).append("\n");
                    for (Line.Info lineInfo : targetLines) {
                        b.append("    - ").append(lineInfo.getLineClass().getSimpleName());
                        b.append(" (").append(lineInfo.toString()).append(")").append("\n");
                    }
                    continue;
                }
                catch (Exception ex) {
                    b.append("  Error accessing mixer: ").append(ex).append("\n");
                }
            }
        }
        catch (Exception ex) {
            b.append("Error getting audio system info: ").append(ex).append("\n");
        }
        b.append("=== End Audio System Debug Information ===").append("\n");
        LOGGER.info(b.toString());
    }
}

