/*
 * Decompiled with CFR 0.152.
 */
package chatty.util.settings;

import chatty.util.DateTime;
import chatty.util.MiscUtil;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class FileManager {
    private static final Logger LOGGER = Logger.getLogger(FileManager.class.getName());
    private static final String MANUAL_BACKUP_PREFIX = "manual_";
    private static final String AUTO_BACKUP_PREFIX = "auto_";
    private static final String SESSION_BACKUP_PREFIX = "session_";
    private static final Charset CHARSET = Charset.forName("UTF-8");
    private final Map<String, FileSettings> files = new HashMap<String, FileSettings>();
    private final Set<String> backupLoaded = new HashSet<String>();
    private final Map<String, String> knownContent = new HashMap<String, String>();
    private final Path basePath;
    private final Path backupPath;
    private boolean savingPaused;
    private static final Pattern FIND_TIMESTAMP = Pattern.compile("auto_([0-9]+)__.+");

    public FileManager(Path basePath, Path backupPath) {
        this.basePath = basePath;
        this.backupPath = backupPath;
    }

    public synchronized void add(String id, String fileName, boolean backupEnabled, FileContentInfoProvider provider) {
        FileSettings settings = new FileSettings(id, this.basePath.resolve(fileName), backupEnabled, provider);
        this.files.put(id, settings);
    }

    public synchronized void setSavingPaused(boolean paused) {
        this.savingPaused = paused;
        LOGGER.info("Saving paused: " + paused);
    }

    public synchronized SaveResult save(String id, String content, boolean force) {
        SaveResult.Builder result = new SaveResult.Builder(id);
        FileSettings fileSettings = this.files.get(id);
        if (this.savingPaused) {
            result.setCancelled(SaveResult.CancelReason.SAVING_PAUSED);
            return result.make();
        }
        if (fileSettings == null) {
            LOGGER.warning("[Save] Invalid file id: " + id);
            result.setCancelled(SaveResult.CancelReason.INVALID_ID);
            return result.make();
        }
        if (!force && this.knownContent.containsKey(id) && Objects.equals(this.knownContent.get(id), content)) {
            if (content != null) {
                LOGGER.info("Not writing " + id + " (known content)");
            }
            result.setCancelled(SaveResult.CancelReason.KNOWN_CONTENT);
            return result.make();
        }
        if (!force && this.backupLoaded.contains(id)) {
            LOGGER.info("Not writing " + id + " (backup loaded this session)");
            result.setCancelled(SaveResult.CancelReason.BACKUP_LOADED);
            return result.make();
        }
        try {
            Path target = fileSettings.path;
            if (content == null) {
                boolean removed = this.removeFile(target);
                if (removed) {
                    result.setRemoved();
                }
            } else {
                this.saveToFile(target, content);
                result.setWritten(target);
            }
            this.knownContent.put(id, content);
        }
        catch (IOException ex) {
            result.setWriteError(ex);
        }
        if (fileSettings.backupEnabled && content != null) {
            try {
                Path backupTarget = this.backupPath.resolve("session__" + fileSettings.path.getFileName());
                this.saveToFile(backupTarget, content);
                result.setBackupWritten(backupTarget);
            }
            catch (IOException ex) {
                result.setBackupError(ex);
            }
        }
        return result.make();
    }

    public synchronized String load(String id) throws IOException {
        FileSettings fileSettings = this.files.get(id);
        if (fileSettings == null) {
            LOGGER.warning("[Load] Invalid file id: " + id);
            return null;
        }
        String content = this.loadFromFile(fileSettings.path);
        this.knownContent.put(id, content);
        return content;
    }

    private String loadFromFile(Path file) throws IOException {
        return new String(Files.readAllBytes(file), CHARSET);
    }

    private void saveToFile(Path file, String content) throws IOException {
        LOGGER.info("Saving contents to file: " + file);
        try {
            Files.createDirectories(file.getParent(), new FileAttribute[0]);
        }
        catch (IOException ex) {
            LOGGER.warning("Failed to create " + file.getParent() + ", let's try writing anyway..");
        }
        try {
            Path tempFile = file.resolveSibling(file.getFileName().toString() + "-temp");
            try (BufferedWriter writer = Files.newBufferedWriter(tempFile, CHARSET, new OpenOption[0]);){
                writer.write(content);
            }
            MiscUtil.moveFile(tempFile, file);
        }
        catch (IOException ex) {
            LOGGER.warning("Error saving file: " + ex);
            throw ex;
        }
    }

    private boolean removeFile(Path file) {
        try {
            Files.delete(file);
            LOGGER.info("Removed unused file: " + file);
            return true;
        }
        catch (NoSuchFileException ex) {
            LOGGER.info("Unused file doesn't exist: " + file);
        }
        catch (IOException ex) {
            LOGGER.warning("Error removing unused file: " + ex);
        }
        return false;
    }

    public synchronized void loadBackup(FileInfo info) throws IOException {
        Files.copy(info.file, ((FileInfo)info).settings.path, StandardCopyOption.REPLACE_EXISTING);
        this.backupLoaded.add(((FileInfo)info).settings.id);
    }

    public synchronized void backup(long backupDelay, int keepCount) throws IOException {
        if (keepCount <= 0) {
            return;
        }
        FileInfos backupFiles = this.getBackupFileInfo();
        long latestTimestamp = backupFiles.getLatestTimestamp(AUTO_BACKUP_PREFIX);
        List<FileInfo> autoFiles = backupFiles.filter(AUTO_BACKUP_PREFIX);
        LOGGER.info(String.format(Locale.ROOT, "[Backup] Latest: %s Delay: %s Count: %d Auto: %d Keep: %d", DateTime.formatFullDatetime(latestTimestamp * 1000L), DateTime.duration(backupDelay * 1000L, 1, 0, new DateTime.Formatting[0]), backupFiles.count(), autoFiles.size(), keepCount));
        int toDelete = Math.min(autoFiles.size() - keepCount, 2);
        for (FileInfo file : autoFiles) {
            if (file.timestamp == -1L || toDelete <= 0) continue;
            try {
                Files.deleteIfExists(file.file);
                LOGGER.info("[Backup] Deleted old backup: " + file.file);
                --toDelete;
            }
            catch (IOException ex) {
                LOGGER.warning("[Backup] Failed to delete backup: " + ex);
            }
        }
        if (System.currentTimeMillis() / 1000L - latestTimestamp > backupDelay) {
            this.doBackup(AUTO_BACKUP_PREFIX);
        }
    }

    public List<SaveResult> manualBackup() {
        return this.doBackup(MANUAL_BACKUP_PREFIX);
    }

    private List<SaveResult> doBackup(String prefix) {
        ArrayList<SaveResult> result = new ArrayList<SaveResult>();
        for (FileSettings file : this.files.values()) {
            result.add(this.doBackup(file, prefix));
        }
        return result;
    }

    private SaveResult doBackup(FileSettings file, String prefix) {
        SaveResult.Builder fileResult = new SaveResult.Builder(file.id);
        if (!file.backupEnabled) {
            fileResult.setCancelled(SaveResult.CancelReason.INVALID);
            return fileResult.make();
        }
        try {
            String content = this.loadFromFile(file.path);
            FileContentInfo info = file.infoProvider.getInfo(content);
            if (!info.isValid) {
                LOGGER.info("[Backup] Didn't perform backup (invalid content): " + file.path);
                fileResult.setCancelled(SaveResult.CancelReason.INVALID_CONTENT);
                return fileResult.make();
            }
            Path backupTarget = this.backupPath.resolve(prefix + System.currentTimeMillis() / 1000L + "__" + file.path.getFileName());
            Files.copy(file.path, backupTarget, StandardCopyOption.REPLACE_EXISTING);
            LOGGER.info("[Backup] Backup performed: " + backupTarget);
            fileResult.setBackupWritten(backupTarget);
        }
        catch (IOException ex) {
            fileResult.setBackupError(ex);
            LOGGER.warning("[Backup] Failed: " + ex);
        }
        return fileResult.make();
    }

    public synchronized Path getBackupPath() {
        return this.backupPath;
    }

    public synchronized FileInfos getBackupFileInfo() throws IOException {
        final ArrayList<FileInfo> result = new ArrayList<FileInfo>();
        HashSet<FileVisitOption> options = new HashSet<FileVisitOption>();
        options.add(FileVisitOption.FOLLOW_LINKS);
        Files.walkFileTree(this.backupPath, options, 1, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                FileSettings s;
                String fileName = file.getFileName().toString();
                String origFileName = FileManager.getOrigFileName(fileName);
                if (FileManager.this.hasValidPrefix(fileName) && origFileName != null && (s = FileManager.this.getFileSettingsByName(origFileName)) != null) {
                    try {
                        String content = FileManager.this.loadFromFile(file);
                        FileContentInfo info = s.infoProvider.getInfo(content);
                        result.add(new FileInfo(s, file, attrs.lastModifiedTime().toMillis(), attrs.size(), FileManager.getTimestamp(fileName), info.isValid, info.info));
                    }
                    catch (IOException ex) {
                        result.add(new FileInfo(s, file, attrs.lastModifiedTime().toMillis(), attrs.size(), FileManager.getTimestamp(fileName), false, "Error: " + ex));
                    }
                }
                return FileVisitResult.CONTINUE;
            }
        });
        Collections.sort(result, (a, b) -> {
            if (((FileInfo)a).modifiedTime > ((FileInfo)b).modifiedTime) {
                return 1;
            }
            if (((FileInfo)a).modifiedTime < ((FileInfo)b).modifiedTime) {
                return -1;
            }
            if (((FileInfo)a).timestamp > ((FileInfo)b).timestamp) {
                return 1;
            }
            if (((FileInfo)a).timestamp < ((FileInfo)b).timestamp) {
                return -1;
            }
            return 0;
        });
        return new FileInfos(result);
    }

    private FileSettings getFileSettingsByName(String fileName) {
        for (FileSettings s : this.files.values()) {
            if (!s.path.getFileName().toString().equals(fileName)) continue;
            return s;
        }
        return null;
    }

    private boolean hasValidPrefix(String fileName) {
        return fileName.startsWith(MANUAL_BACKUP_PREFIX) || fileName.startsWith(AUTO_BACKUP_PREFIX) || fileName.startsWith(SESSION_BACKUP_PREFIX);
    }

    private static String getOrigFileName(String fileName) {
        int index = fileName.indexOf("__");
        if (index == -1 || index + 2 == fileName.length()) {
            return null;
        }
        return fileName.substring(index + 2);
    }

    private static long getTimestamp(String fileName) {
        Matcher m = FIND_TIMESTAMP.matcher(fileName);
        if (m.matches()) {
            return Long.parseLong(m.group(1));
        }
        return -1L;
    }

    public static final void main(String[] args) throws IOException {
        FileManager m = new FileManager(Paths.get("H:\\test123", new String[0]), Paths.get("H:\\test123\\backupp", new String[0]));
        String content = "content\nabc\rblah\r\n";
        m.add("test", "filename", true, new FileContentInfoProvider(){

            @Override
            public FileContentInfo getInfo(String content) {
                if (content != null && !content.isEmpty()) {
                    return new FileContentInfo(true, content.length() + " characters");
                }
                return new FileContentInfo(false, "Empty file");
            }
        });
    }

    public static class FileSettings {
        public final String id;
        public final Path path;
        public final boolean backupEnabled;
        public final FileContentInfoProvider infoProvider;

        public FileSettings(String id, Path path, boolean backupEnabled, FileContentInfoProvider infoProvider) {
            this.id = id;
            this.path = path;
            this.backupEnabled = backupEnabled;
            this.infoProvider = infoProvider;
        }
    }

    public static interface FileContentInfoProvider {
        public FileContentInfo getInfo(String var1);
    }

    public static class SaveResult {
        public final String id;
        public final boolean written;
        public final boolean backupWritten;
        public final Throwable writeError;
        public final Throwable backupError;
        public final Path filePath;
        public final Path backupPath;
        public final boolean removed;
        public final CancelReason cancelReason;

        private SaveResult(Builder builder) {
            this.id = builder.id;
            this.written = builder.written;
            this.backupWritten = builder.backupWritten;
            this.writeError = builder.writeError;
            this.backupError = builder.backupError;
            this.filePath = builder.filePath;
            this.backupPath = builder.backupPath;
            this.removed = builder.removed;
            this.cancelReason = builder.cancelReason;
        }

        private static class Builder {
            private String id;
            private boolean written;
            private boolean backupWritten;
            private Throwable writeError;
            private Throwable backupError;
            private Path filePath;
            private Path backupPath;
            private boolean removed;
            private CancelReason cancelReason;

            public Builder(String id) {
                this.id = id;
            }

            public Builder setWritten(Path path) {
                this.written = true;
                this.filePath = path;
                return this;
            }

            public Builder setBackupWritten(Path path) {
                this.backupWritten = true;
                this.backupPath = path;
                return this;
            }

            public Builder setWriteError(Throwable writeError) {
                this.writeError = writeError;
                return this;
            }

            public Builder setBackupError(Throwable backupError) {
                this.backupError = backupError;
                return this;
            }

            public Builder setRemoved() {
                this.removed = true;
                return this;
            }

            public Builder setCancelled(CancelReason reason) {
                this.cancelReason = reason;
                return this;
            }

            public SaveResult make() {
                return new SaveResult(this);
            }
        }

        public static enum CancelReason {
            BACKUP_LOADED,
            INVALID_ID,
            KNOWN_CONTENT,
            SAVING_PAUSED,
            INVALID_CONTENT,
            INVALID;

        }
    }

    public static class FileInfo {
        private final FileSettings settings;
        private final String info;
        private final long modifiedTime;
        private final boolean isValid;
        private final Path file;
        private final long timestamp;
        private final long size;

        public FileInfo(FileSettings settings, Path file, long modifiedTime, long size, long timestamp, boolean isValid, String info) {
            this.file = file;
            this.modifiedTime = modifiedTime;
            this.info = info;
            this.isValid = isValid;
            this.timestamp = timestamp;
            this.settings = settings;
            this.size = size;
        }

        public FileSettings getSettings() {
            return this.settings;
        }

        public String getInfo() {
            return this.info;
        }

        public boolean isValid() {
            return this.isValid;
        }

        public Path getFile() {
            return this.file;
        }

        public boolean nameStartsWith(String prefix) {
            return this.file.getFileName().toString().startsWith(prefix);
        }

        public long getModifiedTime() {
            return this.modifiedTime;
        }

        public long getSize() {
            return this.size;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        public long getCreated() {
            return this.timestamp == -1L ? this.modifiedTime : this.timestamp * 1000L;
        }

        public String toString() {
            return String.format("[%s] Mod:%s Bu:%s valid: %s (%s)", this.file, DateTime.ago(this.modifiedTime, new DateTime.Formatting[0]), DateTime.ago(this.timestamp * 1000L, new DateTime.Formatting[0]), this.isValid, this.info);
        }
    }

    public static class FileInfos {
        private final List<FileInfo> data;

        public FileInfos(List<FileInfo> data) {
            this.data = data;
        }

        public List<FileInfo> getList() {
            return new ArrayList<FileInfo>(this.data);
        }

        public long getLatestTimestamp(String prefix) {
            long latestTimestamp = 0L;
            for (FileInfo file : this.data) {
                if (file.timestamp <= latestTimestamp || !file.nameStartsWith(prefix)) continue;
                latestTimestamp = file.timestamp;
            }
            return latestTimestamp;
        }

        public List<FileInfo> filter(String prefix) {
            ArrayList<FileInfo> result = new ArrayList<FileInfo>();
            for (FileInfo file : this.data) {
                if (!file.nameStartsWith(prefix)) continue;
                result.add(file);
            }
            return result;
        }

        public int count() {
            return this.data.size();
        }
    }

    public static class FileContentInfo {
        public final boolean isValid;
        public final String info;

        public FileContentInfo(boolean isValid, String info) {
            this.isValid = isValid;
            this.info = info;
        }
    }
}

